/*
	Filename: DecodeAmbi.sc 
	created: 11.4.2005 

	Copyright (C) IEM 2005, Christopher Frauenberger [frauenberger@iem.at] 

	This program is free software; you can redistribute it and/or 
	modify it under the terms of the GNU General Public License 
	as published by the Free Software Foundation; either version 2 
	of the License, or (at your option) any later version. 

	This program is distributed in the hope that it will be useful, 
	but WITHOUT ANY WARRANTY; without even the implied warranty of 
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
	GNU General Public License for more details. 

	You should have received a copy of the GNU General Public License 
	along with this program; if not, write to the Free Software 
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 

	IEM - Institute of Electronic Music and Acoustics, Graz 
	Inffeldgasse 10/3, 8010 Graz, Austria 
	http://iem.at
*/


/* 	Class: DecodeAmbi
	Ambisonics decoder base class for Ambisonics signals up to 5th order
*/
DecodeAmbi { 
	/* 	Class method: *ar
	   	Audio rate method
	   	Parameter
	   		in: Array[] of Ambisonics channels (number depends on order)
			setup: a key (symbol) to the decoding scheme
			level: Level/gain 
		Return
			out: Array[] of loudspeaker signals 
	*/
	*ar { arg in, setup = '1_4_7_4', level=1; 
		var coeffs;
		// check number of in channels match order 
		if (in.size != (this.order + 1).squared, 
			{ "Decode: in has wrong number of channels.".postln; ^nil });
		// get decode matrix 
		coeffs = this.setups[setup]; 
		if (coeffs.isNil, { "Decode: no setup for this name.".postln; ^nil });
		if (coeffs[0].size != in.size, 
			{ "Decode: in and decode matrix dimension don't match.".postln; ^nil });
					 
		^coeffs.collect { |row| (row * in).sum };
	}	
}


/* 	Class: DecodeAmbi2O
	Ambisonics decoder for 2nd order, providing decoding schemes (setups) for 2nd order 
*/
DecodeAmbi2O : DecodeAmbi {  
	// the order of the Ambisonics system
	classvar <order = 2;
	// the Ambisonics decoding scheme used 
	classvar <setups; 
	// the positions of the loudspeakers
	classvar <positions;

	*initClass { 
		setups = (		// identity dictionary 
			// 1(top) 4(raised) 7(earlevel) 4(lowered) loudspeaker setup 
			// LS Setup: 16 speakers
			'1_4_7_4':
			[
				[0.08827,0.00070296,0,0.13175,0.0045255,0,0,0,0.36278],
				[0.056909,0.091767,0.093276,0.13115,-0.00085538,0.13794,0.2141,0.2141,0.028744],
				[0.057302,-0.092465,0.09761,0.13139,-0.0036429,-0.14266,-0.2141,0.2141,0.027326],
				[0.057302,-0.092465,-0.09761,0.13139,-0.0036429,0.14266,-0.2141,-0.2141,0.027326],
				[0.056909,0.091767,-0.093276,0.13115,-0.00085538,-0.13794,0.2141,-0.2141,0.028744],
				[0.044012,0.15452,0.071636,0.025827,0.15698,0.18131,0,0,-0.15893],
				[0.047232,0.041117,0.16835,0.027717,-0.22291,0.11534,0,0,-0.17056],
				[0.046189,-0.1107,0.13861,0.027105,-0.051666,-0.23878,0,0,-0.16679],
				[0.043996,-0.16888,0,0.025818,0.24162,0,0,0,-0.15887],
				[0.046189,-0.1107,-0.13861,0.027105,-0.051666,0.23878,0,0,-0.16679],
				[0.047232,0.041117,-0.16835,0.027717,-0.22291,-0.11534,0,0,-0.17056],
				[0.044012,0.15452,-0.071636,0.025827,0.15698,-0.18131,0,0,-0.15893],
				[0.089998,0.13148,0,-0.21183,0.14858,0,-0.30279,0,0.17433],
				[0.092598,-0.0013568,0.13498,-0.2103,-0.14759,-0.0033351,0,-0.30279,0.16494],
				[0.090553,-0.12907,0,-0.2115,0.14464,0,0.30279,0,0.17232],
				[0.092598,-0.0013568,-0.13498,-0.2103,-0.14759,0.0033351,0,0.30279,0.16494]
			], 
			// CUBE setup 
			'4_8_12': []			
		);
		
		positions = (
			// 1(top) 4(raised) 7(earlevel) 4(lowered) loudspeaker setup 
			// LS Setup: 16 speakers
			'1_4_7_4':
			[ 
				[90,40,40,40,40,0,0,0,0,0,0,0,-40,-40,-40,-40], 
				[0,45,135,225,315,25,75,130,180,230,285,335,0,90,180,270]
			]
		);

	}
}

/* 	Class: DecodeAmbi3O
	Ambisonics decoder for 3rd order, providing decoding schemes (setups)
*/
DecodeAmbi3O : DecodeAmbi {  
	classvar <order = 3;
	// the Ambisonics decoding scheme used 
	classvar <setups; 
	// the positions of the loudspeakers
	classvar <positions;
	*initClass { 
		setups = (		// identity dictionary 
			// 1(top) 4(raised) 7(earlevel) 4(lowered) loudspeaker setup 
			// LS Setup: 16 speakers
			'1_4_7_4':
			[
[0, 0, 0, 0.322101, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0.726768],
[0.100976, 0.113014, 0.113014, 0.036663, 0, 0, 0.214103, 0.214103, 0.208336, 0, 0, 0, 0.364916, 0.194488, 0.194488, -0.361204],
[0.100976, -0.113014, 0.113014, 0.036663, 0, 0, -0.214103, 0.214103, 0.208336, 0, 0, 0, -0.364916, -0.194488, 0.194488, -0.361204],
[0.100976, -0.113014, -0.113014, 0.036663, 0, 0, -0.214103, -0.214103, 0.208336, 0, 0, 0, 0.364916, -0.194488, -0.194488, -0.361204],
[0.100976, 0.113014, -0.113014, 0.036663, 0, 0, 0.214103, -0.214103, 0.208336, 0, 0, 0, -0.364916, 0.194488, -0.194488, -0.361204],
[0.026595, 0.132079, 0.051495, 0.062879, 0.215830, 0.270101, 0.035593, -0.160550, -0.229082, 0.088168, 0.397699, 0.154997, -0.193971, -0.213249, -0.083142, 0.141875],
[0.028216, 0.034125, 0.153929, 0.066710, -0.298839, 0.139351, -0.092610, 0.120691, -0.243042, -0.229404, -0.298965, -0.214609, -0.100074, -0.055097, -0.248527, 0.150520],
[0.029476, -0.094457, 0.118208, 0.069691, -0.098693, -0.324723, 0.136411, -0.063609, -0.253900, 0.337903, 0.157567, -0.070876, 0.233198, 0.152506, -0.190853, 0.157246],
[0.024918, -0.143495, 0, 0.058914, 0.363403, 0, -0.158788, 0, -0.214637, -0.393333, 0, 0.260976, 0, 0.231681, 0, 0.132929],
[0.029476, -0.094457, -0.118208, 0.069691, -0.098693, 0.324723, 0.136411, 0.063609, -0.253900, 0.337903, -0.157567, -0.070876, -0.233198, 0.152506, 0.190853, 0.157246],
[0.028216, 0.034125, -0.153929, 0.066710, -0.298839, -0.139351, -0.092610, -0.120691, -0.243042, -0.229404, 0.298965, -0.214609, 0.100074, -0.055097, 0.248527, 0.150520],
[0.026595, 0.132079, -0.051495, 0.062879, 0.215830, -0.270101, 0.035593, 0.160550, -0.229082, 0.088168, -0.397699, 0.154997, 0.193971, -0.213249, 0.083142, 0.141875],
[0.100976, 0.159826, 0, -0.231557, 0, 0, -0.302787, 0, 0.208336, 0, 0, -0.364916, 0, 0.275048, 0, -0.078541],
[0.100976, 0, 0.159826, -0.231557, 0, 0, 0, -0.302787, 0.208336, 0, 0, 0.364916, 0, 0, 0.275048, -0.078541],
[0.100976, -0.159826, 0, -0.231557, 0, 0, 0.302787, 0, 0.208336, 0, 0, -0.364916, 0, -0.275048, 0, -0.078541],
[0.100976, 0, -0.159826, -0.231557, 0, 0, 0, 0.302787, 0.208336, 0, 0, 0.364916, 0, 0, -0.275048, -0.078541],
			], 
			// CUBE setup 
			// three cube setups for 25 loudspeaker (25th speaker is virtual and at the top of the sphere - 
			// its signal should be multiplied by 0.25 and shared by the last four real speakers)
			'CUBE_strong': [ 
[ 0.0625, 0.1549, 0, -0.0349, 0.123, 0, -0.0521, 0, -0.0506, 0.045, 0, -0.0214, 0, -0.0157, 0, 0.0103 ], 
[ 0.0625, 0.1301, -0.0704, -0.0349, 0.0615, -0.1035, -0.0458, 0.0271, -0.0506, -0.0002, -0.0453, -0.0107, 0.0148, -0.0161, 0.0123, 0.0103 ], 
[ 0.0625, 0.0704, -0.1301, -0.0349, -0.0615, -0.1035, -0.0271, 0.0458, -0.0506, -0.0453, -0.0002, 0.0107, 0.0148, -0.0123, 0.0161, 0.0103 ], 
[ 0.0625, 0, -0.1549, -0.0349, -0.123, 0, 0, 0.0521, -0.0506, 0, 0.045, 0.0214, 0, 0, 0.0157, 0.0103 ], 
[ 0.0625, -0.0704, -0.1301, -0.0349, -0.0615, 0.1035, 0.0271, 0.0458, -0.0506, 0.0453, -0.0002, 0.0107, -0.0148, 0.0123, 0.0161, 0.0103 ], 
[ 0.0625, -0.1301, -0.0704, -0.0349, 0.0615, 0.1035, 0.0458, 0.0271, -0.0506, 0.0002, -0.0453, -0.0107, -0.0148, 0.0161, 0.0123, 0.0103 ], 
[ 0.0625, -0.1549, 0, -0.0349, 0.123, 0, 0.0521, 0, -0.0506, -0.045, 0, -0.0214, 0, 0.0157, 0, 0.0103 ], 
[ 0.0625, -0.1301, 0.0704, -0.0349, 0.0615, -0.1035, 0.0458, -0.0271, -0.0506, 0.0002, 0.0453, -0.0107, 0.0148, 0.0161, -0.0123, 0.0103 ], 
[ 0.0625, -0.0704, 0.1301, -0.0349, -0.0615, -0.1035, 0.0271, -0.0458, -0.0506, 0.0453, 0.0002, 0.0107, 0.0148, 0.0123, -0.0161, 0.0103 ], 
[ 0.0625, 0, 0.1549, -0.0349, -0.123, 0, 0, -0.0521, -0.0506, 0, -0.045, 0.0214, 0, 0, -0.0157, 0.0103 ], 
[ 0.0625, 0.0704, 0.1301, -0.0349, -0.0615, 0.1035, -0.0271, -0.0458, -0.0506, -0.0453, 0.0002, 0.0107, -0.0148, -0.0123, -0.0161, 0.0103 ], 
[ 0.0625, 0.1301, 0.0704, -0.0349, 0.0615, 0.1035, -0.0458, -0.0271, -0.0506, -0.0002, 0.0453, -0.0107, -0.0148, -0.0161, -0.0123, 0.0103 ],
[ 0.0454, 0.0914, -0.0357, 0.0579, 0.0516, -0.0475, 0.08, -0.0328, -0.0103, 0.0081, -0.017, 0.0255, -0.02, 0.0094, -0.0025, -0.017 ], 
[ 0.0454, 0.0357, -0.0914, 0.0579, -0.0516, -0.0475, 0.0328, -0.08, -0.0103, -0.017, 0.0081, -0.0255, -0.02, 0.0025, -0.0094, -0.017 ],
[ 0.0454, -0.0357, -0.0914, 0.0579, -0.0516, 0.0475, -0.0328, -0.08, -0.0103, 0.017, 0.0081, -0.0255, 0.02, -0.0025, -0.0094, -0.017 ],
[ 0.0454, -0.0914, -0.0357, 0.0579, 0.0516, 0.0475, -0.08, -0.0328, -0.0103, -0.0081, -0.017, 0.0255, 0.02, -0.0094, -0.0025, -0.017 ],
[ 0.0454, -0.0914, 0.0357, 0.0579, 0.0516, -0.0475, -0.08, 0.0328, -0.0103, -0.0081, 0.017, 0.0255, -0.02, -0.0094, 0.0025, -0.017 ], 
[ 0.0454, -0.0357, 0.0914, 0.0579, -0.0516, -0.0475, -0.0328, 0.08, -0.0103, 0.017, -0.0081, -0.0255, -0.02, -0.0025, 0.0094, -0.017 ], 
[ 0.0454, 0.0357, 0.0914, 0.0579, -0.0516, 0.0475, 0.0328, 0.08, -0.0103, -0.017, -0.0081, -0.0255, 0.02, 0.0025, 0.0094, -0.017 ], 
[ 0.0454, 0.0914, 0.0357, 0.0579, 0.0516, 0.0475, 0.08, 0.0328, -0.0103, 0.0081, 0.017, 0.0255, 0.02, 0.0094, 0.0025, -0.017 ], 
[ 0.0449, 0.0523, -0.0523, 0.0954, 0, -0.0246, 0.0614, -0.0614, 0.0625, -0.0008, -0.0008, 0, -0.0157, 0.0286, -0.0286, 0.0106 ], 
[ 0.0449, -0.0523, -0.0523, 0.0954, 0, 0.0246, -0.0614, -0.0614, 0.0625, 0.0008, -0.0008, 0, 0.0157, -0.0286, -0.0286, 0.0106 ], 
[ 0.0449, -0.0523, 0.0523, 0.0954, 0, -0.0246, -0.0614, 0.0614, 0.0625, 0.0008, 0.0008, 0, -0.0157, -0.0286, 0.0286, 0.0106 ], 
[ 0.0449, 0.0523, 0.0523, 0.0954, 0, 0.0246, 0.0614, 0.0614, 0.0625, -0.0008, 0.0008, 0, 0.0157, 0.0286, 0.0286, 0.0106 ], 
[ 0.0414, 0, 0, 0.1092, 0, 0, 0, 0, 0.0997, 0, 0, 0, 0, 0, 0, 0.0348 ]
		],
		'CUBE_weak':	[ 
[ 0.0453, 0.1315, 0, -0.0296, 0.1782, 0, -0.0756, 0, -0.0733, 0.1002, 0, -0.0477, 0, -0.0349, 0, 0.0228 ], 
[ 0.0453, 0.1104, -0.0597, -0.0296, 0.0891, -0.1499, -0.0663, 0.0393, -0.0733, -0.0003, -0.1008, -0.0238, 0.0329, -0.036, 0.0274, 0.0228 ], 
[ 0.0453, 0.0597, -0.1104, -0.0296, -0.0891, -0.1499, -0.0393, 0.0663, -0.0733, -0.1008, -0.0003, 0.0238, 0.0329, -0.0274, 0.036, 0.0228 ], 
[ 0.0453, 0, -0.1315, -0.0296, -0.1782, 0, 0, 0.0756, -0.0733, 0, 0.1002, 0.0477, 0, 0, 0.0349, 0.0228 ], 
[ 0.0453, -0.0597, -0.1104, -0.0296, -0.0891, 0.1499, 0.0393, 0.0663, -0.0733, 0.1008, -0.0003, 0.0238, -0.0329, 0.0274, 0.036, 0.0228 ], 
[ 0.0453, -0.1104, -0.0597, -0.0296, 0.0891, 0.1499, 0.0663, 0.0393, -0.0733, 0.0003, -0.1008, -0.0238, -0.0329, 0.036, 0.0274, 0.0228 ], 
[ 0.0453, -0.1315, 0, -0.0296, 0.1782, 0, 0.0756, 0, -0.0733, -0.1002, 0, -0.0477, 0, 0.0349, 0, 0.0228 ],
[ 0.0453, -0.1104, 0.0597, -0.0296, 0.0891, -0.1499, 0.0663, -0.0393, -0.0733, 0.0003, 0.1008, -0.0238, 0.0329, 0.036, -0.0274, 0.0228 ], 
[ 0.0453, -0.0597, 0.1104, -0.0296, -0.0891, -0.1499, 0.0393, -0.0663, -0.0733, 0.1008, 0.0003, 0.0238, 0.0329, 0.0274, -0.036, 0.0228 ], 
[ 0.0453, 0, 0.1315, -0.0296, -0.1782, 0, 0, -0.0756, -0.0733, 0, -0.1002, 0.0477, 0, 0, -0.0349, 0.0228 ], 
[ 0.0453, 0.0597, 0.1104, -0.0296, -0.0891, 0.1499, -0.0393, -0.0663, -0.0733, -0.1008, 0.0003, 0.0238, -0.0329, -0.0274, -0.036, 0.0228 ], 
[ 0.0453, 0.1104, 0.0597, -0.0296, 0.0891, 0.1499, -0.0663, -0.0393, -0.0733, -0.0003, 0.1008, -0.0238, -0.0329, -0.036, -0.0274, 0.0228 ],
[ 0.0329, 0.0776, -0.0303, 0.0491, 0.0747, -0.0688, 0.116, -0.0476, -0.015, 0.0181, -0.0378, 0.0568, -0.0445, 0.0209, -0.0056, -0.0379 ], 
[ 0.0329, 0.0303, -0.0776, 0.0491, -0.0747, -0.0688, 0.0476, -0.116, -0.015, -0.0378, 0.0181, -0.0568, -0.0445, 0.0056, -0.0209, -0.0379 ], 
[ 0.0329, -0.0303, -0.0776, 0.0491, -0.0747, 0.0688, -0.0476, -0.116, -0.015, 0.0378, 0.0181, -0.0568, 0.0445, -0.0056, -0.0209, -0.0379 ], 
[ 0.0329, -0.0776, -0.0303, 0.0491, 0.0747, 0.0688, -0.116, -0.0476, -0.015, -0.0181, -0.0378, 0.0568, 0.0445, -0.0209, -0.0056, -0.0379 ], 
[ 0.0329, -0.0776, 0.0303, 0.0491, 0.0747, -0.0688, -0.116, 0.0476, -0.015, -0.0181, 0.0378, 0.0568, -0.0445, -0.0209, 0.0056, -0.0379 ], 
[ 0.0329, -0.0303, 0.0776, 0.0491, -0.0747, -0.0688, -0.0476, 0.116, -0.015, 0.0378, -0.0181, -0.0568, -0.0445, -0.0056, 0.0209, -0.0379 ], 
[ 0.0329, 0.0303, 0.0776, 0.0491, -0.0747, 0.0688, 0.0476, 0.116, -0.015, -0.0378, -0.0181, -0.0568, 0.0445, 0.0056, 0.0209, -0.0379 ], 
[ 0.0329, 0.0776, 0.0303, 0.0491, 0.0747, 0.0688, 0.116, 0.0476, -0.015, 0.0181, 0.0378, 0.0568, 0.0445, 0.0209, 0.0056, -0.0379 ], 
[ 0.0325, 0.0444, -0.0444, 0.081, 0, -0.0357, 0.089, -0.089, 0.0906, -0.0018, -0.0018, 0, -0.035, 0.0636, -0.0636, 0.0236 ], 
[ 0.0325, -0.0444, -0.0444, 0.081, 0, 0.0357, -0.089, -0.089, 0.0906, 0.0018, -0.0018, 0, 0.035, -0.0636, -0.0636, 0.0236 ], 
[ 0.0325, -0.0444, 0.0444, 0.081, 0, -0.0357, -0.089, 0.089, 0.0906, 0.0018, 0.0018, 0, -0.035, -0.0636, 0.0636, 0.0236 ], 
[ 0.0325, 0.0444, 0.0444, 0.081, 0, 0.0357, 0.089, 0.089, 0.0906, -0.0018, 0.0018, 0, 0.035, 0.0636, 0.0636, 0.0236 ], 
[ 0.03, 0, 0, 0.0927, 0, 0, 0, 0, 0.1445, 0, 0, 0, 0, 0, 0, 0.0776 ] 
		],
		'CUBE_basic': [ 
[ 0.0391, 0.1134, 0, -0.0255, 0.1537, 0, -0.0652, 0, -0.0632, 0.192, 0, -0.0913, 0, -0.0668, 0, 0.0438 ], 
[ 0.0391, 0.0952, -0.0515, -0.0255, 0.0769, -0.1293, -0.0572, 0.0339, -0.0632, -0.0006, -0.1931, -0.0457, 0.063, -0.0689, 0.0526, 0.0438 ], 
[ 0.0391, 0.0515, -0.0952, -0.0255, -0.0769, -0.1293, -0.0339, 0.0572, -0.0632, -0.1931, -0.0006, 0.0457, 0.063, -0.0526, 0.0689, 0.0438 ], 
[ 0.0391, 0, -0.1134, -0.0255, -0.1537, 0, 0, 0.0652, -0.0632, 0, 0.192, 0.0913, 0, 0, 0.0668, 0.0438 ], 
[ 0.0391, -0.0515, -0.0952, -0.0255, -0.0769, 0.1293, 0.0339, 0.0572, -0.0632, 0.1931, -0.0006, 0.0457, -0.063, 0.0526, 0.0689, 0.0438 ],
[ 0.0391, -0.0952, -0.0515, -0.0255, 0.0769, 0.1293, 0.0572, 0.0339, -0.0632, 0.0006, -0.1931, -0.0457, -0.063, 0.0689, 0.0526, 0.0438 ], 
[ 0.0391, -0.1134, 0, -0.0255, 0.1537, 0, 0.0652, 0, -0.0632, -0.192, 0, -0.0913, 0, 0.0668, 0, 0.0438 ], 
[ 0.0391, -0.0952, 0.0515, -0.0255, 0.0769, -0.1293, 0.0572, -0.0339, -0.0632, 0.0006, 0.1931, -0.0457, 0.063, 0.0689, -0.0526, 0.0438 ], 
[ 0.0391, -0.0515, 0.0952, -0.0255, -0.0769, -0.1293, 0.0339, -0.0572, -0.0632, 0.1931, 0.0006, 0.0457, 0.063, 0.0526, -0.0689, 0.0438 ], 
[ 0.0391, 0, 0.1134, -0.0255, -0.1537, 0, 0, -0.0652, -0.0632, 0, -0.192, 0.0913, 0, 0, -0.0668, 0.0438 ], 
[ 0.0391, 0.0515, 0.0952, -0.0255, -0.0769, 0.1293, -0.0339, -0.0572, -0.0632, -0.1931, 0.0006, 0.0457, -0.063, -0.0526, -0.0689, 0.0438 ], 
[ 0.0391, 0.0952, 0.0515, -0.0255, 0.0769, 0.1293, -0.0572, -0.0339, -0.0632, -0.0006, 0.1931, -0.0457, -0.063, -0.0689, -0.0526, 0.0438 ],
[ 0.0284, 0.0669, -0.0262, 0.0424, 0.0645, -0.0593, 0.1, -0.041, -0.0129, 0.0347, -0.0725, 0.1089, -0.0853, 0.04, -0.0108, -0.0727 ], 
[ 0.0284, 0.0262, -0.0669, 0.0424, -0.0645, -0.0593, 0.041, -0.1, -0.0129, -0.0725, 0.0347, -0.1089, -0.0853, 0.0108, -0.04, -0.0727 ], 
[ 0.0284, -0.0262, -0.0669, 0.0424, -0.0645, 0.0593, -0.041, -0.1, -0.0129, 0.0725, 0.0347, -0.1089, 0.0853, -0.0108, -0.04, -0.0727 ], 
[ 0.0284, -0.0669, -0.0262, 0.0424, 0.0645, 0.0593, -0.1, -0.041, -0.0129, -0.0347, -0.0725, 0.1089, 0.0853, -0.04, -0.0108, -0.0727 ], 
[ 0.0284, -0.0669, 0.0262, 0.0424, 0.0645, -0.0593, -0.1, 0.041, -0.0129, -0.0347, 0.0725, 0.1089, -0.0853, -0.04, 0.0108, -0.0727 ], 
[ 0.0284, -0.0262, 0.0669, 0.0424, -0.0645, -0.0593, -0.041, 0.1, -0.0129, 0.0725, -0.0347, -0.1089, -0.0853, -0.0108, 0.04, -0.0727 ], 
[ 0.0284, 0.0262, 0.0669, 0.0424, -0.0645, 0.0593, 0.041, 0.1, -0.0129, -0.0725, -0.0347, -0.1089, 0.0853, 0.0108, 0.04, -0.0727 ], 
[ 0.0284, 0.0669, 0.0262, 0.0424, 0.0645, 0.0593, 0.1, 0.041, -0.0129, 0.0347, 0.0725, 0.1089, 0.0853, 0.04, 0.0108, -0.0727 ], 
[ 0.028, 0.0383, -0.0383, 0.0699, 0, -0.0308, 0.0767, -0.0767, 0.0782, -0.0034, -0.0034, 0, -0.0671, 0.122, -0.122, 0.0452 ], 
[ 0.028, -0.0383, -0.0383, 0.0699, 0, 0.0308, -0.0767, -0.0767, 0.0782, 0.0034, -0.0034, 0, 0.0671, -0.122, -0.122, 0.0452 ], 
[ 0.028, -0.0383, 0.0383, 0.0699, 0, -0.0308, -0.0767, 0.0767, 0.0782, 0.0034, 0.0034, 0, -0.0671, -0.122, 0.122, 0.0452 ], 
[ 0.028, 0.0383, 0.0383, 0.0699, 0, 0.0308, 0.0767, 0.0767, 0.0782, -0.0034, 0.0034, 0, 0.0671, 0.122, 0.122, 0.0452 ], 
[ 0.0259, 0, 0, 0.0799, 0, 0, 0, 0, 0.1246, 0, 0, 0, 0, 0, 0, 0.1487 ] 
		]
		);
		// positions are [ [elevation], [azimuth] ]
		positions = (
			// 1(top) 4(raised) 7(earlevel) 4(lowered) loudspeaker setup 
			// LS Setup: 16 speakers
			'1_4_7_4':
			[ 
				[90,40,40,40,40,0,0,0,0,0,0,0,-40,-40,-40,-40], 
				[0,45,135,225,315,25,75,130,180,230,285,335,0,90,180,270]
			],
			// three settings for the CUBE (25th loudspeaker is virtual and at the top of the sphere)
			// all three are the same!
			'CUBE_strong':
			[ 
				[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 30, 30, 30, 30, 30, 30, 30, 60, 60, 60, 60, 90], 
				[0, 330, 300, 270, 240, 210, 180, 150, 120, 90, 60, 30, 337.5, 292.5, 247.5, 202.5, 157.5, 112.5, 67.5, 22.5, 315, 225, 135, 45, 0]
			], 
			'CUBE_weak':
			[ 
				[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 30, 30, 30, 30, 30, 30, 30, 60, 60, 60, 60, 90], 
				[0, 330, 300, 270, 240, 210, 180, 150, 120, 90, 60, 30, 337.5, 292.5, 247.5, 202.5, 157.5, 112.5, 67.5, 22.5, 315, 225, 135, 45, 0]
			], 
			'CUBE_basic':
			[ 
				[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 30, 30, 30, 30, 30, 30, 30, 30, 60, 60, 60, 60, 90], 
				[0, 330, 300, 270, 240, 210, 180, 150, 120, 90, 60, 30, 337.5, 292.5, 247.5, 202.5, 157.5, 112.5, 67.5, 22.5, 315, 225, 135, 45, 0]
			]

		);

	}

}
/* 	Class: DecodeAmbi4O
	Ambisonics decoder for 4th order, providing decoding schemes (setups)
*/
DecodeAmbi4O : DecodeAmbi {  
	classvar <order = 4;
}
/* 	Class: DecodeAmbi5O
	Ambisonics decoder for 5th order, providing decoding schemes (setups)
*/
DecodeAmbi5O : DecodeAmbi {  
	classvar <order = 5;
}


